/*
 *	Interface GTK+ Theme Engine
 *
 *  interface_theme_patches.c:
 *		Run-time modifications to GTK+.
 *
 *	Copyright  2000 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "Interface_theme.h"
#include <stddef.h>		/* for offsetof() */

/* Exported functions */
void interface_patches_install ();
void interface_patches_remove ();

#define PATCH_INSTALL(klass, type, name, field) \
	interface_patch_install ((GtkObjectClass *)klass, offsetof(type, field), \
						(GtkFunction)interface_ ## name ## _ ## field, (GtkFunction *)&old_ ## name ## _ ## field)

#define PATCH_RESTORE(klass, type, name, field) \
	interface_patch_restore ((GtkObjectClass *)klass, offsetof(type, field), \
						(GtkFunction)interface_ ## name ## _ ## field, (GtkFunction)old_ ## name ## _ ## field)

void interface_patch_install (GtkObjectClass *klass, size_t offset,
						 GtkFunction interface_func, GtkFunction * old_func_ptr);

void interface_patch_restore (GtkObjectClass *klass, size_t offset,
						 GtkFunction interface_func, GtkFunction old_func);


/* Patches */
static void interface_hscrollbar_size_request (GtkWidget *, GtkRequisition *);
static void interface_vscrollbar_size_request (GtkWidget *, GtkRequisition *);
static void interface_vscrollbar_size_allocate (GtkWidget *, GtkAllocation *);
static void interface_hscrollbar_size_allocate (GtkWidget *, GtkAllocation *);
static void interface_vscrollbar_realize (GtkWidget *);
static void interface_hscrollbar_realize (GtkWidget *);
static void interface_vscrollbar_slider_update (GtkRange *);
static void interface_hscrollbar_slider_update (GtkRange *);
static gint interface_vscrollbar_button_press_event (GtkWidget *, GdkEventButton *);
static gint interface_hscrollbar_button_press_event (GtkWidget *, GdkEventButton *);
static gint interface_vscrollbar_trough_click (GtkRange *, gint, gint, gfloat *);
static gint interface_hscrollbar_trough_click (GtkRange *, gint, gint, gfloat *);

static void interface_vscale_draw_trough (GtkRange *);
static void interface_hscale_draw_trough (GtkRange *);
static void interface_vscale_draw_slider (GtkRange *);
static void interface_hscale_draw_slider (GtkRange *);

static gint interface_vscale_expose_event(GtkWidget *, GdkEventExpose *);
static void interface_vscale_draw_focus (GtkWidget *);
static gint interface_hscale_expose_event(GtkWidget *, GdkEventExpose *);
static void interface_hscale_draw_focus (GtkWidget *);

static void interface_vpaned_realize (GtkWidget *);
static void interface_hpaned_realize (GtkWidget *);
static void interface_vpaned_size_allocate(GtkWidget *, GtkAllocation *);
static void interface_hpaned_size_allocate(GtkWidget *, GtkAllocation *);

/* static void interface_combo_realize (GtkWidget *);
static void interface_combo_size_allocate(GtkWidget *, GtkAllocation *); */
/* static void interface_entry_realize (GtkWidget *);
static void interface_entry_size_allocate(GtkWidget *, GtkAllocation *); */

/* static void interface_clist_realize (GtkWidget *); */

/* Original (or previous) functions and values */
void (* old_vscrollbar_realize)(GtkWidget *);
void (* old_hscrollbar_realize)(GtkWidget *);
void (* old_vscrollbar_slider_update)(GtkRange *);
void (* old_hscrollbar_slider_update)(GtkRange *);
void (*	old_vscrollbar_size_request)(GtkWidget *, GtkRequisition *);
void (*	old_hscrollbar_size_request)(GtkWidget *, GtkRequisition *);
void (*	old_vscrollbar_size_allocate)(GtkWidget *, GtkAllocation *);
void (*	old_hscrollbar_size_allocate)(GtkWidget *, GtkAllocation *);
gint (* old_vscrollbar_button_press_event)(GtkWidget *, GdkEventButton *);
gint (* old_hscrollbar_button_press_event)(GtkWidget *, GdkEventButton *);
gint (* old_vscrollbar_trough_click)(GtkRange *, gint, gint, gfloat *);
gint (* old_hscrollbar_trough_click)(GtkRange *, gint, gint, gfloat *);

void (* old_vpaned_size_allocate)(GtkWidget *, GtkAllocation *);
void (* old_hpaned_size_allocate)(GtkWidget *, GtkAllocation *);
void (* old_vpaned_realize)(GtkWidget *);
void (* old_hpaned_realize)(GtkWidget *);

void (* old_hscale_draw_trough)(GtkRange *);
void (* old_hscale_draw_slider)(GtkRange *);
gint (* old_hscale_expose_event)(GtkWidget *, GdkEventExpose *);
void (* old_hscale_draw_focus)(GtkWidget *);

void (* old_vscale_draw_trough)(GtkRange *);
void (* old_vscale_draw_slider)(GtkRange *);
gint (* old_vscale_expose_event)(GtkWidget *, GdkEventExpose *);
void (* old_vscale_draw_focus)(GtkWidget *);

/* void (* old_entry_realize)(GtkWidget *);
void (* old_entry_size_allocate)(GtkWidget *, GtkAllocation *); */
/* void (* old_combo_realize)(GtkWidget *);
void (* old_combo_size_allocate)(GtkWidget *, GtkAllocation *); */

/* void (* old_clist_realize)(GtkWidget *); */

gint old_scrollbar_spacing;
gint old_stepper_slider_spacing;
guint16 old_check_size;
guint16 old_radio_size;


/*
	Install and Remove
*/

void interface_patches_install () {
	GtkWidgetClass	*widget_class;

	/* Make scrollbars wider */
	{
		GtkRangeClass	*range_class;
		
		range_class = (GtkRangeClass *)gtk_type_class(gtk_scrollbar_get_type());

		old_stepper_slider_spacing = range_class->stepper_slider_spacing;
		range_class->stepper_slider_spacing = 0;
	}

	/* Move the scrollbar buttons to the lower right corner */
	{
		widget_class = (GtkWidgetClass *) gtk_type_class(gtk_vscrollbar_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscrollbar, size_request);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscrollbar, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscrollbar, realize);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscrollbar, button_press_event);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscrollbar, slider_update);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscrollbar, trough_click);

		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscrollbar_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscrollbar, size_request);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscrollbar, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscrollbar, realize);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscrollbar, button_press_event);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscrollbar, slider_update);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscrollbar, trough_click);
	}
	
	/* Make radio and check buttons larger */
	{
		GtkCheckButtonClass *checkbuttonclass;
		
		checkbuttonclass = (GtkCheckButtonClass *) gtk_type_class (gtk_radio_button_get_type ());
		old_radio_size = checkbuttonclass->indicator_size;
		checkbuttonclass->indicator_size = INTERFACE_RADIO_SIZE;
		
		checkbuttonclass = (GtkCheckButtonClass *) gtk_type_class (gtk_check_button_get_type ());
		old_check_size = checkbuttonclass->indicator_size;
		checkbuttonclass->indicator_size = INTERFACE_CHECK_SIZE;
	}

	/* Make ScrolledWindow spacing smaller */
	{
		GtkScrolledWindowClass *scwinclass;
		
		scwinclass = (GtkScrolledWindowClass *) gtk_type_class (gtk_scrolled_window_get_type ());
		old_scrollbar_spacing = scwinclass->scrollbar_spacing;
		scwinclass->scrollbar_spacing = 1;
	}

	/* Make entire panes sensitive */
	{
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vpaned_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, vpaned, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vpaned, realize);
		
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hpaned_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, hpaned, size_allocate);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hpaned, realize);
	}

	/* Draw sliders differently */
	{
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscale_get_type());
		PATCH_INSTALL(widget_class, GtkRangeClass, hscale, draw_trough);
		PATCH_INSTALL(widget_class, GtkRangeClass, hscale, draw_slider);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscale, expose_event);
		PATCH_INSTALL(widget_class, GtkWidgetClass, hscale, draw_focus);

		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vscale_get_type());
		PATCH_INSTALL(widget_class, GtkRangeClass, vscale, draw_trough);
		PATCH_INSTALL(widget_class, GtkRangeClass, vscale, draw_slider);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscale, expose_event);
		PATCH_INSTALL(widget_class, GtkWidgetClass, vscale, draw_focus);
	}

	/* Combo box / entry combination */
	{
/*		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_combo_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, combo, realize);
		PATCH_INSTALL(widget_class, GtkWidgetClass, combo, size_allocate); */

/*		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_entry_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, entry, realize);
		PATCH_INSTALL(widget_class, GtkWidgetClass, entry, size_allocate); */
	}

	/* CList */
/*	{
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_clist_get_type());
		PATCH_INSTALL(widget_class, GtkWidgetClass, clist, realize);
	} */
}


void interface_patches_remove () {
	GtkWidgetClass	*widget_class;

	/* CList */
/*	{
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_clist_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, clist, realize);
	} */

	/* GtkCombo, GtkEntry */
	{
/*		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_entry_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, entry, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, entry, realize); */
		
/*		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_combo_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, combo, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, combo, realize); */
	}
			
	/* GtkVScale, GtkHScale */
	{
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vscale_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscale, draw_focus);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscale, expose_event);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscale, draw_slider);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscale, draw_trough);

		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscale_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscale, draw_focus);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscale, expose_event);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscale, draw_slider);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscale, draw_trough);
	}

	/* GtkVPaned, GtkHPaned.. they suck btw, use my new paned table class and be happy!
	   You can find it at <http://www.obsession.se/johan/gtk.html>.
	*/
	{
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hpaned_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, hpaned, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hpaned, size_allocate);
	
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vpaned_get_type());
		PATCH_RESTORE(widget_class, GtkWidgetClass, vpaned, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vpaned, size_allocate);
	}

	/* GtkScrolledWindow */
	{
		GtkScrolledWindowClass *scwinclass;
		
		scwinclass = (GtkScrolledWindowClass *) gtk_type_class (gtk_scrolled_window_get_type ());
		scwinclass->scrollbar_spacing = old_scrollbar_spacing;
	}
	
	/* GtkRadioButton and GtkCheckButton */
	{
		GtkCheckButtonClass *checkbuttonclass;

		checkbuttonclass = (GtkCheckButtonClass *) gtk_type_class (gtk_check_button_get_type ());
		checkbuttonclass->indicator_size = old_check_size;
	
		checkbuttonclass = (GtkCheckButtonClass *) gtk_type_class (gtk_radio_button_get_type ());
		checkbuttonclass->indicator_size = old_radio_size;
	}

	/* Scrollbars */
	{
		GtkRangeClass	*range_class;
	
		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_hscrollbar_get_type());
		PATCH_RESTORE(widget_class, GtkRangeClass, hscrollbar, trough_click);
		PATCH_RESTORE(widget_class, GtkRangeClass, hscrollbar, slider_update);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscrollbar, button_press_event);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscrollbar, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscrollbar, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, hscrollbar, size_request);

		widget_class = (GtkWidgetClass *)gtk_type_class(gtk_vscrollbar_get_type());
		PATCH_RESTORE(widget_class, GtkRangeClass, vscrollbar, trough_click);
		PATCH_RESTORE(widget_class, GtkRangeClass, vscrollbar, slider_update);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscrollbar, button_press_event);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscrollbar, realize);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscrollbar, size_allocate);
		PATCH_RESTORE(widget_class, GtkWidgetClass, vscrollbar, size_request);
		
		range_class = (GtkRangeClass *)gtk_type_class(gtk_scrollbar_get_type());
		range_class->stepper_slider_spacing = old_stepper_slider_spacing;
	}
}


/*
 *	Patch Control
 */

void interface_patch_install (GtkObjectClass *klass, size_t offset,
						 GtkFunction interface_func, GtkFunction * old_func_ptr)
{
  /*
	GList			*children, *child;
  */
	GtkFunction	*ptr;

	ptr = (GtkFunction *)((guchar *)klass + offset);

	*old_func_ptr = *ptr;
	if (*ptr == interface_func) {
		g_warning ("Redundant patch of %s::%d", gtk_type_name(klass->type), offset);
	}
	
	*ptr = interface_func;
}


void interface_patch_restore (GtkObjectClass *klass, size_t offset,
						 GtkFunction interface_func, GtkFunction old_func)
{
	GList		*child;
	GtkFunction	*ptr;

	ptr = (GtkFunction *)((guchar *)klass + offset);
	if (*ptr == interface_func)
		*ptr = old_func;

	for (child = gtk_type_children_types(klass->type); child; child = child->next)
		interface_patch_restore (gtk_type_class(GPOINTER_TO_UINT(child->data)),
							offset, interface_func, old_func);
}



/*
 * Patched functions
 */
#define RANGE_CLASS(widget) \
	GTK_RANGE_CLASS(GTK_OBJECT(widget)->klass)


static void
interface_scrollbar_config (GtkWidget *widget, gint *y, gint *h, gint *sb, gint *sf, gint *sp, gint c) {
	InterfaceThemeData	*data;
	gint			s, t, yt;

	if (   widget->style->klass == &interface_class
		|| widget->style->klass == &interface_thin_class)
		data = (InterfaceThemeData *)(widget->style->engine_data);
	else
		data = NULL;

	s	= data ? data->scrollbar_width : RANGE_CLASS(widget)->stepper_size;
	if (s<8) s = RANGE_CLASS(widget)->stepper_size;
	t	= RANGE_CLASS(widget)->stepper_slider_spacing;
	
	*h -= s+s+t;
	
	if (data && data->stepper_ends==FALSE) {
		if (   (widget->parent)
			&& GTK_IS_SCROLLED_WINDOW(widget->parent)
			&& (GTK_SCROLLED_WINDOW(widget->parent)->window_placement & c))
		{
			if (sb) *sb = *y;
			*y += s+s+t;
		} else {
			if (sb) *sb = *y + *h + t;
		}
		if (sf) *sf = *sb+s;
	} else {
		if (sb) *sb = *y;
		if (sf) *sf = *y + s + *h + t;
		*y += s;
		*h -= t;
	}
	if (sp) *sp = s;
}


/* GtkVScrollbar */
static gint
interface_vscrollbar_trough_click (GtkRange *range, gint x, gint y, gfloat *jump_perc) {
	gint	xt, ty, tw, th, sy, sh;

	g_return_val_if_fail (range != NULL, GTK_TROUGH_NONE);
	g_return_val_if_fail (GTK_IS_VSCROLLBAR (range), GTK_TROUGH_NONE);

	xt = ((GtkWidget *)range)->style->klass->xthickness;
	if (x < xt)
		return GTK_TROUGH_NONE;
	
	ty = ((GtkWidget *)range)->style->klass->ythickness;
	gdk_window_get_size (range->trough, &tw, &th);
	th -= ty+ty;

	if (x >= tw-xt)
		return GTK_TROUGH_NONE;
	
	interface_scrollbar_config ((GtkWidget *)range, &ty, &th, NULL, NULL, NULL, 0x01);
	if (jump_perc) {
		*jump_perc = ((gdouble)(y-ty)) / ((gdouble)th);
		return GTK_TROUGH_JUMP;
	} else {
		gdk_window_get_position (range->slider, NULL, &sy);
		if (y < sy)
			return GTK_TROUGH_START;
		else
			return GTK_TROUGH_END;
	}
	return GTK_TROUGH_NONE;
}


static void
interface_vscrollbar_size_request (GtkWidget *widget, GtkRequisition *requisition) {
	InterfaceThemeData	*data;
	gint w;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (widget));
	g_return_if_fail (requisition != NULL);

	if (   widget->style->klass == &interface_class
		|| widget->style->klass == &interface_thin_class)
		data = (InterfaceThemeData *)(widget->style->engine_data);
	else
		data = NULL;

	w = data ? data->scrollbar_width : RANGE_CLASS(widget)->slider_width;
	requisition->width	= widget->style->klass->xthickness * 2 + w;
	requisition->height	= (widget->style->klass->ythickness + w) * 2 + w;
	widget->requisition = *requisition;
}

static void
interface_vscrollbar_realize (GtkWidget *widget) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (widget));

	old_vscrollbar_realize (widget);
	interface_vscrollbar_size_allocate (widget, &(widget->allocation));
}

static void
interface_vscrollbar_slider_update(GtkRange *range) {
	GtkAdjustment *adjustment;
	gint	top, bot;
	gint	x,y,w,h, ty,tw, th, xt,yt, s;
	gfloat	ah;

	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (range));

	if (GTK_WIDGET_REALIZED((GtkWidget *)range)) {
		gdk_window_get_geometry (range->trough, NULL, NULL, &w, &h, NULL);
		xt = ((GtkWidget *)range)->style->klass->xthickness;
		yt = ((GtkWidget *)range)->style->klass->ythickness;

		x   = xt;
		y   = yt;
		w  -= xt+xt;
		h  -= yt+yt;
		interface_scrollbar_config((GtkWidget *)range, &y, &h, NULL, NULL, &s, 0x01);
		ty	 = y;
		th   = h;
		
		adjustment = range->adjustment;
		ah = adjustment->upper - adjustment->lower;
		
		if (adjustment->page_size > 0 && ah > 1.0E-5) {
			if (adjustment->page_size > ah)
				adjustment->page_size = ah;
			
			h = th * range->adjustment->page_size / ah;
			if (h < s)
				h = s;
			
			if (ah - adjustment->page_size != 0)
				y += (th-h) * (adjustment->value - adjustment->lower) / (ah - adjustment->page_size);
		}
		if (y<ty) y=ty;
		
		gdk_window_move_resize (range->slider, x, y, w, h);
	}
}

static void
interface_vscrollbar_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkRange	*range;
	gint 		w, s, x, y, ty,th, sb,sf, e;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VSCROLLBAR (widget));
	g_return_if_fail (allocation != NULL);

	range = GTK_RANGE(widget);

	widget->allocation = *allocation;
	if (GTK_WIDGET_REALIZED (widget)) {
		x = widget->style->klass->xthickness;
		y = widget->style->klass->ythickness;
		ty = y;
		th = widget->allocation.height - y * 2;
		interface_scrollbar_config(widget, &ty, &th, &sb, &sf, &s, 0x01);

		w = widget->allocation.width - x * 2;
		e = (w - s)/2;

		gdk_window_move_resize (range->step_back, x, sb, s, s);
		gdk_window_move_resize (range->step_forw, x, sf, s, s);
		gdk_window_move_resize (range->trough,
								allocation->x+e,		allocation->y,
								s+x*2,					allocation->height);

		RANGE_CLASS(widget)->slider_update(GTK_RANGE(widget));
	}
}

static gint interface_vscrollbar_button_press_event (GtkWidget *widget, GdkEventButton *event) {
	GdkEventButton	fake_event;
	GtkRange		*range;
	gint			offset;
	
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_VSCROLLBAR (widget), FALSE);
	range = GTK_RANGE(widget);

	fake_event = *event;
	if (   range->button == 0
		&& event->window == range->slider)
	{
		gint	y=0, h=0;
		interface_scrollbar_config(widget, &y, &h, NULL, NULL, &offset, 0x01);
		offset += RANGE_CLASS(widget)->stepper_slider_spacing - y;
		if (event->window == range->slider)
			fake_event.y -= offset;
	}
	return GTK_WIDGET_CLASS(gtk_type_class(gtk_scrollbar_get_type()))->
			button_press_event(widget, &fake_event);
}


/* GtkHScrollbar */
static gint
interface_hscrollbar_trough_click (GtkRange *range, gint x, gint y, gfloat *jump_perc) {
	gint	yt, tx, tw, th, sx, sw;

	g_return_val_if_fail (range != NULL, GTK_TROUGH_NONE);
	g_return_val_if_fail (GTK_IS_HSCROLLBAR (range), GTK_TROUGH_NONE);

	yt = ((GtkWidget *)range)->style->klass->ythickness;
	if (y < yt)
		return GTK_TROUGH_NONE;
	
	tx = ((GtkWidget *)range)->style->klass->xthickness;
	gdk_window_get_size (range->trough, &tw, &th);
	tw -= tx+tx;

	if (y >= th-yt)
		return GTK_TROUGH_NONE;
	
	interface_scrollbar_config ((GtkWidget *)range, &tx, &tw, NULL, NULL, NULL, 0x02);

	if (jump_perc) {
		*jump_perc = ((gdouble)(x-tx)) / ((gdouble)tw);
		return GTK_TROUGH_JUMP;
	} else {
		gdk_window_get_position (range->slider, &sx, NULL);
		if (x < sx)
			return GTK_TROUGH_START;
		else
			return GTK_TROUGH_END;
	}
	return GTK_TROUGH_NONE;
}


static void
interface_hscrollbar_size_request (GtkWidget *widget, GtkRequisition *requisition) {
	InterfaceThemeData	*data;
	gint			h;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (widget));
	g_return_if_fail (requisition != NULL);

	if (   widget->style->klass == &interface_class
		|| widget->style->klass == &interface_thin_class)
		data = (InterfaceThemeData *)(widget->style->engine_data);
	else
		data = NULL;

	h = data ? data->scrollbar_width : RANGE_CLASS(widget)->slider_width;
	requisition->width	= (widget->style->klass->xthickness + h) * 2 + h;
	requisition->height	= widget->style->klass->ythickness * 2 + h;
	widget->requisition = *requisition;
}

static void
interface_hscrollbar_realize (GtkWidget *widget) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (widget));

	old_hscrollbar_realize (widget);
	interface_hscrollbar_size_allocate (widget, &(widget->allocation));
}

static void
interface_hscrollbar_slider_update(GtkRange *range) {
	GtkAdjustment *adjustment;
	gint x,y,w,h, tx,tw, xt, yt, s;
	gfloat aw;

	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (range));

	if (GTK_WIDGET_REALIZED((GtkWidget *)range)) {
		gdk_window_get_geometry (range->trough, NULL, NULL, &w, &h, NULL);
		xt = ((GtkWidget *)range)->style->klass->xthickness;
		yt = ((GtkWidget *)range)->style->klass->ythickness;

		x   = xt;
		y   = yt;
		w  -= xt+xt;
		h  -= yt+yt;
		interface_scrollbar_config((GtkWidget *)range, &x, &w, NULL, NULL, &s, 0x02);
		tw   = w;
		tx	 = x;

		adjustment = range->adjustment;
		aw = adjustment->upper - adjustment->lower;
		
		if (adjustment->page_size > 0 && aw > 1.0E-5) {
			if (adjustment->page_size > aw)
				adjustment->page_size = aw;

			w = tw * range->adjustment->page_size / aw;
			if (w < s)
				w = s;
			
			if (aw - adjustment->page_size != 0)
				x += (tw-w) * (adjustment->value - adjustment->lower) / (aw - adjustment->page_size);
		}
		if (x<xt) x=xt;
		
		gdk_window_move_resize (range->slider, x, y, w, h);
	}
}

static void
interface_hscrollbar_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkRange *range;
	gint h, s, x, y, tx,tw, sb,sf, e;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HSCROLLBAR (widget));
	g_return_if_fail (allocation != NULL);

	range = GTK_RANGE (widget);
	
	widget->allocation = *allocation;
	if (GTK_WIDGET_REALIZED (widget)) {
		x = widget->style->klass->xthickness;
		y = widget->style->klass->ythickness;
		tx = x;
		tw = widget->allocation.width - x * 2;
		interface_scrollbar_config(widget, &tx, &tw, &sb, &sf, &s, 0x02);

		h = widget->allocation.height - y * 2;
		e = (h - s)/2;
		
		gdk_window_move_resize (range->step_back, sb, y, s, s);
		gdk_window_move_resize (range->step_forw, sf, y, s, s);
		
		gdk_window_move_resize (range->trough,
								allocation->x,		allocation->y+e,
								allocation->width,	s+y*2);
		RANGE_CLASS(range)->slider_update(range);
	}
}

static gint interface_hscrollbar_button_press_event (GtkWidget *widget, GdkEventButton *event) {
	GdkEventButton	fake_event;
	GtkRange		*range;
	gint			offset;
	
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_HSCROLLBAR (widget), FALSE);
	range = GTK_RANGE(widget);

	fake_event = *event;
	if (range->button == 0) {
		gint	x=0, w=0;
		
		interface_scrollbar_config(widget, &x, &w, NULL, NULL, &offset, 0x02);
		offset += RANGE_CLASS(widget)->stepper_slider_spacing - x;

		if (event->window == range->slider)
			fake_event.x -= offset;
	}
	return GTK_WIDGET_CLASS(gtk_type_class(gtk_scrollbar_get_type()))->
			button_press_event(widget, &fake_event);
}


/* Panes */
static void interface_hpaned_realize (GtkWidget *widget) {
	GdkCursor *crsr;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HPANED(widget));

	old_hpaned_realize(widget);
	crsr = gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);
	gdk_window_set_cursor(GTK_PANED(widget)->handle, crsr);
	gdk_cursor_destroy(crsr);
	
	interface_hpaned_size_allocate (widget, &(widget->allocation));
}

static void interface_hpaned_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkPaned	*paned;
	gint		x, width;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_HPANED(widget));

	paned = ((GtkPaned*)widget);
	paned->handle_size = paned->gutter_size;

	old_hpaned_size_allocate (widget, allocation);
	if (GTK_WIDGET_REALIZED(widget)) {
		gdk_window_get_geometry(paned->handle, &x, NULL, &width, NULL, NULL);
		gdk_window_move_resize (paned->handle,
								x,
								0,
								paned->handle_size,
								widget->allocation.height);
	}
	gtk_widget_queue_draw(widget);
}

static void interface_vpaned_realize (GtkWidget *widget) {
	GdkCursor *crsr;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VPANED(widget));

	old_vpaned_realize(widget);
	crsr = gdk_cursor_new(GDK_SB_V_DOUBLE_ARROW);
	gdk_window_set_cursor(GTK_PANED(widget)->handle, crsr);
	gdk_cursor_destroy(crsr);
	interface_vpaned_size_allocate (widget, &(widget->allocation));
}

static void interface_vpaned_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkPaned	*paned;
	gint		y, height;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_VPANED(widget));

	paned = ((GtkPaned*)widget);
	paned->handle_size = paned->gutter_size;

	old_vpaned_size_allocate (widget, allocation);
	if (GTK_WIDGET_REALIZED(widget)) {
		gdk_window_get_geometry(paned->handle, NULL, &y, NULL, &height, NULL);
		gdk_window_move_resize (paned->handle,
								0,
								y,
								widget->allocation.width,
								paned->handle_size);
	}
	gtk_widget_queue_draw(widget);
}

/* GtkScale */

static gint interface_hscale_expose_event(GtkWidget *widget, GdkEventExpose *event) {
	interface_vscale_expose_event(widget, event);
}

static gint interface_vscale_expose_event(GtkWidget *widget, GdkEventExpose *event) {
	GtkRange *range;
	
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SCALE (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	
	range = GTK_RANGE (widget);
	
	if (event->window == range->trough) {
		/* no declining to draw */
		gtk_range_draw_trough(range);
	} else if (event->window == range->slider) {
		gtk_range_draw_slider(range);
	} else if (event->window == range->step_forw) {
		gtk_range_draw_step_forw(range);
	} else if (event->window == range->step_back) {
		gtk_range_draw_step_back(range);
	} else if (event->window == widget->window) {
		gtk_range_draw_background(range);
	}
	return FALSE;
}

static void interface_hscale_draw_focus (GtkWidget *widget) {
	interface_vscale_draw_focus(widget);
}

static void interface_vscale_draw_focus (GtkWidget *widget) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SCALE (widget));
	
	if (GTK_WIDGET_DRAWABLE (widget))
		gtk_range_draw_slider (GTK_RANGE (widget));
}


/* GtkHScale */
static void interface_hscale_draw_trough (GtkRange *range) {
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_HSCALE (range));
	
	if (range->trough) {
		GtkStyle *style;
		int trough_y, trough_width, trough_height, xt, yt;

		gdk_window_get_size (range->trough, &trough_width, &trough_y);
		style = ((GtkWidget *)range)->style;
		xt = style->klass->xthickness;
		yt = style->klass->ythickness;
		
		trough_height = (trough_y & 0x01) + yt+yt + 2;
		trough_y  = trough_y/2 - yt - 1;
		
		gtk_paint_flat_box (((GtkWidget *)range)->parent->style, range->trough,
							((GtkWidget *)range)->parent->state, GTK_SHADOW_NONE, NULL,
							(GtkWidget *)range, "scale trough", 0, 0, -1, -1);
		gtk_paint_box (style, range->trough,
					   ((GtkWidget*)range)->state==GTK_STATE_INSENSITIVE
					   ? GTK_STATE_INSENSITIVE
					   : GTK_STATE_ACTIVE,
					   GTK_SHADOW_IN, NULL,
					   (GtkWidget *)range, "trough", xt, trough_y, trough_width-xt-xt, trough_height);
		
		if (range->slider) {
			int	slider_x;
			gdk_window_get_position(range->slider, &slider_x, NULL);
			if (slider_x > (xt+xt)) {
				gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], NULL);
				gdk_draw_rectangle (range->trough, style->bg_gc[GTK_STATE_SELECTED], TRUE,
									xt+xt, trough_y + yt,
									slider_x - (xt+xt), trough_height - yt - yt);
			}
		}
	}
}

static void interface_hscale_draw_slider (GtkRange *range) {
	GtkStateType state_type;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_HSCALE (range));
	
	if (range->slider) {
		if ((range->in_child == RANGE_CLASS (range)->slider) ||
			(range->click_child == RANGE_CLASS (range)->slider))
			state_type = GTK_STATE_PRELIGHT;
		else if (GTK_WIDGET_STATE(range) == GTK_STATE_INSENSITIVE)
			state_type = GTK_STATE_INSENSITIVE;
		else
			state_type = GTK_STATE_NORMAL;
		
		gtk_paint_slider (GTK_WIDGET (range)->style, range->slider, state_type, 
						  GTK_SHADOW_OUT, NULL, GTK_WIDGET (range), "hscale",
						  0, 0, -1, -1, GTK_ORIENTATION_HORIZONTAL);
	}
}

/* GtkVScale */

static void interface_vscale_draw_trough (GtkRange *range) {
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_VSCALE (range));
	
	if (range->trough) {
		GtkStyle *style;
		int trough_x, trough_width, trough_height, xt, yt;

		gdk_window_get_size (range->trough, &trough_x, &trough_height);
		style = ((GtkWidget *)range)->style;
		xt = style->klass->xthickness;
		yt = style->klass->ythickness;
		
		trough_width = (trough_x & 0x01) + xt+xt + 2;
		trough_x  = trough_x/2 - xt - 1;
		
		gtk_paint_flat_box (((GtkWidget *)range)->parent->style, range->trough,
							((GtkWidget *)range)->parent->state, GTK_SHADOW_NONE, NULL,
							(GtkWidget *)range, "scale trough", 0, 0, -1, -1);
		gtk_paint_box (style, range->trough,
					   ((GtkWidget*)range)->state==GTK_STATE_INSENSITIVE
					   ? GTK_STATE_INSENSITIVE
					   : GTK_STATE_ACTIVE,
					   GTK_SHADOW_IN, NULL,
					   (GtkWidget *)range, "trough", trough_x, yt, trough_width, trough_height-yt-yt);
		
		if (range->slider) {
			int	slider_y;
			gdk_window_get_position(range->slider, NULL, &slider_y);
			if (slider_y > yt+yt) {
				gdk_gc_set_clip_rectangle(style->bg_gc[GTK_STATE_SELECTED], NULL);
				gdk_draw_rectangle (range->trough, style->bg_gc[GTK_STATE_SELECTED],
									TRUE, 
									trough_x + xt, yt+yt,
									trough_width - xt-xt, slider_y - yt-yt);
			}
		}
	}
}

static void interface_vscale_draw_slider (GtkRange *range) {
	GtkStateType state_type;
	
	g_return_if_fail (range != NULL);
	g_return_if_fail (GTK_IS_VSCALE (range));
	
	if (range->slider) {
		if ((range->in_child == RANGE_CLASS (range)->slider) ||
			(range->click_child == RANGE_CLASS (range)->slider))
			state_type = GTK_STATE_PRELIGHT;
		else if (GTK_WIDGET_STATE(range) == GTK_STATE_INSENSITIVE)
			state_type = GTK_STATE_INSENSITIVE;
		else
			state_type = GTK_STATE_NORMAL;

		gtk_paint_slider (GTK_WIDGET (range)->style, range->slider, state_type,
						  GTK_SHADOW_OUT, NULL, GTK_WIDGET (range), "hscale",
						  0, 0, -1, -1, GTK_ORIENTATION_VERTICAL);
	}
}

/* static void interface_combo_realize (GtkWidget *widget) {
	old_combo_realize (widget);
}

static void interface_combo_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	GtkCombo		*combo;
	GtkAllocation	child_allocation;
	gushort			xt, yt;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (allocation != NULL);
	g_return_if_fail (GTK_IS_COMBO(widget));

	old_combo_size_allocate(widget, allocation);

	combo = GTK_COMBO(widget);
	xt = widget->style->klass->xthickness;
	yt = widget->style->klass->ythickness;

	child_allocation.height = combo->entry->requisition.height - yt - yt;
	child_allocation.width	 = child_allocation.height;
	child_allocation.x		 = allocation->x
							 + allocation->width
							 - child_allocation.width
							 - ((GtkContainer *)widget)->border_width
							 - xt;
	child_allocation.y		 = combo->entry->allocation.y
							 + (combo->entry->allocation.height - combo->entry->requisition.height) / 2
							 + yt;

	gtk_widget_size_allocate(combo->button, &child_allocation);

	child_allocation.x		 = combo->entry->allocation.x;
	child_allocation.y		 = combo->entry->allocation.y;
	child_allocation.width	 = allocation->width - ((GtkContainer *)widget)->border_width * 2;
	child_allocation.height	 = allocation->height - ((GtkContainer *)widget)->border_width * 2;

	gtk_widget_size_allocate (combo->entry, &child_allocation);
} */


/* static void interface_entry_resize (GtkWidget *widget) {
	GtkEntry	*entry;
	gushort		text_area_width;
	gint		xoffset, max_offset;
	
	entry = GTK_ENTRY(widget);
	text_area_width	= widget->allocation.width
					- widget->requisition.height
					+ widget->style->klass->ythickness * 2
					- widget->style->klass->xthickness * 2;
	
	gdk_window_resize (GTK_ENTRY(widget)->text_area,
					   text_area_width,
					   widget->requisition.height - widget->style->klass->ythickness * 2);
	
	text_area_width -= widget->style->klass->xthickness;
	max_offset = entry->char_offset[entry->text_length] - text_area_width;
	if (max_offset < 0)
		max_offset = 0;
	if (entry->scroll_offset > max_offset)
		entry->scroll_offset = max_offset;
	
	xoffset = entry->char_offset[GTK_EDITABLE(entry)->current_pos];
	xoffset -= entry->scroll_offset;
	
	if (xoffset < 0)
		entry->scroll_offset += xoffset;
	else if (xoffset > text_area_width)
		entry->scroll_offset += xoffset - text_area_width + 1;
}

static void interface_entry_realize (GtkWidget *widget) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_ENTRY(widget));

	old_entry_realize(widget);

	if (   widget->parent
		&& GTK_IS_COMBO(widget->parent))
	{
		interface_entry_resize(widget);
	}
}

static void interface_entry_size_allocate (GtkWidget *widget, GtkAllocation *allocation) {
	g_return_if_fail (widget != NULL);
	g_return_if_fail (allocation != NULL);
	g_return_if_fail (GTK_IS_ENTRY(widget));

	old_entry_size_allocate(widget, allocation);

	if (   widget->parent
		&& GTK_IS_COMBO(widget->parent)
		&& GTK_WIDGET_REALIZED(widget))
	{
		interface_entry_resize(widget);
	}
} */

/* GtkCList */

/* static void interface_clist_realize (GtkWidget *widget) {
	GdkGCValues	values;
	GdkGC		*gc;
	
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_CLIST (widget));

	old_clist_realize (widget);

	values.foreground = widget->style->base[GTK_STATE_SELECTED];
	values.function = GDK_XOR;
	values.subwindow_mode = GDK_INCLUDE_INFERIORS;

	gc = gdk_gc_new_with_values (widget->window, &values, GDK_GC_FOREGROUND | GDK_GC_FUNCTION | GDK_GC_SUBWINDOW);
	if (gc) {
		gdk_gc_unref(GTK_CLIST(widget)->xor_gc);
		GTK_CLIST(widget)->xor_gc = gc;
	}
} */


